﻿using CursoApi.Dtos;
using CursoApi.Services;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;

namespace CursoApi.Controllers
{
    [Route("rural-properties")]
    [ApiController]
    [Authorize]
    public class RuralPropertiesController : ControllerBase
    {
        IRuralPropertyService _propertyService;
        IWeatherForecastService _forecastService;

        public RuralPropertiesController(IRuralPropertyService propertyService, IWeatherForecastService forecastService)
        {
            _propertyService = propertyService;
            _forecastService = forecastService;
        }

        /// <summary>
        /// Lista todas as propriedades de um usuário.
        /// </summary>
        /// <returns></returns>
        [HttpGet]
        public async Task<ActionResult<IList<RuralPropertyDto>>> Get()
        {
            return Ok(await _propertyService.List());
        }

        /// <summary>
        /// Consulta uma propriedade pelo Id.
        /// </summary>
        /// <param name="id">Id da propriedade.</param>
        /// <returns></returns>
        [HttpGet("{id}")]
        [ProducesResponseType(StatusCodes.Status200OK)]
        [ProducesResponseType(StatusCodes.Status404NotFound)]
        public async Task<ActionResult<RuralPropertyDto>> Get(int id)
        {
            var result = await _propertyService.Get(id);
            return result != null ? Ok(await _propertyService.Get(id)) : NotFound();
        }

        /// <summary>
        /// Insere uma nova propriedade.
        /// </summary>
        /// <param name="dto"></param>
        /// <returns></returns>
        [HttpPost]
        [ProducesResponseType(StatusCodes.Status201Created)]
        [ProducesResponseType(StatusCodes.Status400BadRequest)]
        public async Task<ActionResult<RuralPropertyDto>> Post(RuralPropertyDto dto)
        {
            var result = await _propertyService.Insert(dto);
            return CreatedAtAction(nameof(Get), new { Id = result.Id }, result);
        }

        /// <summary>
        /// Edita uma propriedade rural.
        /// </summary>
        /// <param name="dto"></param>
        /// <returns></returns>
        [HttpPatch]
        [ProducesResponseType(StatusCodes.Status204NoContent)]
        [ProducesResponseType(StatusCodes.Status400BadRequest)]
        public async Task<IActionResult> Patch(RuralPropertyDto dto)
        {
            await _propertyService.Edit(dto);
            return NoContent();
        }

        /// <summary>
        /// Excluí uma propriedade.
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        [HttpDelete("{id}")]
        [ProducesResponseType(StatusCodes.Status204NoContent)]
        public async Task<IActionResult> Delete(int id)
        {
            await _propertyService.Delete(id);
            return NoContent();
        }

        /// <summary>
        /// Retorna relatório de previsão meteorológica para determinada propriedade.
        /// </summary>
        /// <param name="id">Id da propriedade.</param>
        /// <returns></returns>
        [HttpGet("{id}/weather-report")]
        public async Task<ActionResult<WeatherForecastDto>> GetForecast(int id)
        {
            return Ok(await _forecastService.Get(id));
        }
    }
}
